# Public Directory

This directory contains **static assets** that are served directly by Vite without server-side processing. Files in this directory are **publicly accessible** and served as if they were in the root folder of the application.

## Vite Public Directory Behavior

The `public` directory follows **Vite's static asset handling**:

- **Root-level serving** - Files are served from the root URL path
- **No bundling** - Assets are copied as-is, not processed by Vite
- **Direct URL access** - Files accessible via direct URL paths
- **Build-time copying** - Files copied to the output directory during build

## Import Guidelines

**CRITICAL: Do NOT import from "public" folder in code**

```typescript
// ❌ FORBIDDEN: Importing from public folder
import logo from "/public/images/WorkBook_logo.png";

// ✅ CORRECT: Import from root path (Vite serves from root)
import logo from "/images/WorkBook_logo.png";

// ✅ CORRECT: Reference in HTML/templates
<img src="/images/WorkBook_logo.png" alt="WorkBook Logo" />

// ✅ CORRECT: CSS background images
.logo {
    background-image: url("/images/WorkBook_logo.png");
}
```

## Directory Structure

### `/Fonts/` - Typography Assets
- **wbfont.otf** - WorkBook custom font (OpenType)
- **wbfont.woff** - Web font format for cross-browser compatibility
- **wbfont.fcp** - Font configuration/project file

### `/images/` - Visual Assets

#### `/images/PNG/` - Bitmap Images
Large collection of PNG images including:
- **UI Icons** - Approval status, navigation, system icons
- **State Indicators** - Error, warning, success, status icons
- **User Interface Elements** - Checkboxes, buttons, controls
- **Theme Assets** - jQuery UI theme images and gradients
- **Brand Assets** - App store badges, platform icons

#### `/images/SVG/` - Vector Graphics
Comprehensive SVG icon library:
- **Business Icons** - Jobs, tasks, resources, finances
- **Navigation Icons** - Arrows, menus, controls
- **Status Icons** - Approval states, progress indicators
- **Action Icons** - Edit, delete, add, save operations
- **Functional Icons** - Calendar, search, filters, settings

#### `/images/controls/` - UI Control Assets
Color-coded control images for different themes:
- **Theme Colors** - Blue, green, red, orange variations
- **High-DPI Support** - @2x retina images included
- **Dropdown Controls** - Select box styling assets

#### `/images/login/` - Authentication Assets
Login screen branding:
- **WorkBook Logos** - Various sizes and formats
- **Brand Assets** - Transparent and standard versions

#### `/images/main/` - Application Assets
Main application icons and imagery:
- **App Icons** - Different resolutions (256px, 512px, 1024px)
- **Interface Icons** - Search, refresh controls

#### `/images/message/` - Notification Assets
Message box and notification icons:
- **Status Icons** - Error, warning, info, question, success
- **Multiple Sizes** - 16px, 24px, 32px, 64px variants
- **Consistent Design** - Matching visual style across sizes

### `/sounds/` - Audio Assets
Notification and interaction sounds:
- **System Sounds** - Notification, success, error sounds
- **UI Feedback** - Button clicks, pop-up sounds
- **Message Alerts** - New message notifications

### `/vendors/` - Third-Party Assets
Static third-party files not managed by NPM:
- Legacy vendor assets requiring direct file serving
- Third-party resources with specific URL requirements

### Root Files

#### Web Application Manifest
- **manifest.json** - PWA configuration for mobile/desktop app behavior

#### SEO and Crawling
- **robots.txt** - Search engine crawler instructions

#### Application Entry
- **app_online.htm** - Application entry point or fallback page

#### Branding Assets
- **workbook.ico** - Favicon for browser tabs
- **workbook-*.svg** - Logo variations (horizontal, vertical, white versions)

## Asset Organization Guidelines

### Adding New Assets

**File Naming Conventions:**
- Use **lowercase** with underscores or hyphens
- Include **size indicators** in filenames (e.g., `icon_16.png`)
- Use **descriptive names** that indicate purpose

**SVG Icon Workflow:**
1. **Add SVG files** to `/images/SVG/` directory
2. **Run build script** - Execute `npm run svg:build` to optimize icons and generate output files
3. **Commit changes** - Include both source SVG and generated files

**Directory Organization:**
```
/images/
  /category/        # Group by functional category
    icon_16.png     # Size indicator in filename
    icon@2x.png     # High-DPI variant
```

### Image Optimization

**PNG Images:**
- **Optimize file sizes** using image compression tools
- **Include @2x variants** for high-DPI displays
- **Consistent sizing** - use standard dimensions (16px, 24px, 32px, 64px)

**SVG Icons:**
- **Run build script** - After adding new SVG icons, run `npm run svg:build` to optimize and generate output files
- **Optimize SVG code** - remove unnecessary metadata
- **Consistent viewBox** - use standard viewBox dimensions
- **Semantic naming** - clear, descriptive filenames

### Performance Considerations

**Caching:**
- Files are **cached by browsers** - use cache-busting for updates
- **Long-term caching** - assets rarely change, optimize for caching

**Loading:**
- **Lazy loading** - load images only when needed
- **Preload critical assets** - important images for first paint
- **WebP alternatives** - consider modern formats for better compression

### Security Considerations

**Public Access:**
- All files are **publicly accessible** via direct URL
- **No sensitive data** should be stored in public directory
- **Sanitize filenames** - avoid special characters or system paths

**File Types:**
- **Whitelist allowed extensions** - only serve safe file types
- **Scan uploaded assets** - if allowing user uploads to public directory

### Build Integration

**Vite Processing:**
- Files are **copied as-is** to build output
- **No bundling or optimization** by Vite
- **Preserve directory structure** in build output

**Production Deployment:**
- Files served by **web server directly**
- Configure **appropriate MIME types**
- Set **caching headers** for optimal performance

## Usage Patterns

### In Vue Components

```vue
<template>
    <img src="/images/login/WorkBook_logo.png" alt="WorkBook" />
    <audio ref="notification">
        <source src="/sounds/NotificationSound.wav" type="audio/wav" />
    </audio>
</template>

<script setup lang="ts">
// Access via root paths, not /public/ prefix
const logoUrl = "/images/login/WorkBook_logo.png";
</script>
```

### In CSS/SCSS

```css
.app-logo {
    background: url("/images/login/WorkBook_logo.png") no-repeat;
    background-size: contain;
}

.notification-icon {
    content: url("/images/PNG/Info_16.png");
}
```

### Dynamic Asset Loading

```typescript
// Correct way to load public assets dynamically
const loadIcon = (iconName: string) => {
    return `/images/SVG/${iconName}.svg`;
};

// Usage
const approvalIcon = loadIcon("ApprovalStatus");
```

---

*This directory serves static assets directly to users without processing, following Vite's public directory conventions for optimal performance and caching.*
